#!/usr/bin/env python
# -*- coding: utf-8 -*-

from base64 import b64encode, b64decode
from cPickle import dumps, loads
from uuid import uuid4

from gluon import current
from gluon.http import HTTP


__all__ = ["create_confirmation_code", "validate_code",
           "validate_code_from_request"]


_CODE_KEY = "conf-{}-{}"
_VALUE_KEY = "value-{}-{}"
_EXPIRATION = 60*60  # One hour.


def create_confirmation_code(service, value):
    """
    `service` must be a string to identify the service where the code
    will be used.
    """
    # Create a random code.
    code = str(uuid4())
    # Store the value.
    current.cache.disk(
        _VALUE_KEY.format(service, code),
        lambda: b64encode(dumps(value)),
        _EXPIRATION
    )
    return code


def validate_code(service, code):
    """
    Check if `code` is valid. If it is, the associated value is returned
    and the code is invalidated. Otherwise, `None` is returned.
    """
    # Retrieve the associated value.
    value_key = _VALUE_KEY.format(service, code)
    value = current.cache.disk(value_key, lambda: None, _EXPIRATION)
    # Remove it once used.
    current.cache.disk(value_key, None)
    if value is not None:
        return loads(b64decode(value))


def validate_code_from_request(service):
    """
    Sames as `validate_code()`, but the code itself is take from
    request.args(0). If it is None or invalid, 404 is raised.
    """
    code = current.request.args(0)
    if code is None:
        raise HTTP(404)
    storage = validate_code(service, code)
    if storage is None:
        raise HTTP(404)
    return storage

